# 1 "ocamldoc/odoc_ocamlhtml.mll"
 
(**************************************************************************)
(*                                                                        *)
(*                                 OCaml                                  *)
(*                                                                        *)
(*             Maxence Guesdon, projet Cristal, INRIA Rocquencourt        *)
(*                                                                        *)
(*   Copyright 2001 Institut National de Recherche en Informatique et     *)
(*     en Automatique.                                                    *)
(*                                                                        *)
(*   All rights reserved.  This file is distributed under the terms of    *)
(*   the GNU Lesser General Public License version 2.1, with the          *)
(*   special exception on linking described in the file LICENSE.          *)
(*                                                                        *)
(**************************************************************************)

(** Generation of html code to display OCaml code. *)
open Lexing

type error =
  | Illegal_character of char
  | Unterminated_comment
  | Unterminated_string
  | Keyword_as_label of string

exception Error of error * int * int

let base_escape_strings = [
    ("&", "&amp;") ;
    ("<", "&lt;") ;
    (">", "&gt;") ;
]


let prelike_escape_strings = [
  (" ", "&nbsp;") ;
  ("\t", "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;") ;
  ("\n", "<br>\n")
]


let pre = ref false
let fmt = ref Format.str_formatter

(** Escape the strings which would clash with html syntax,
   and some other strings if we want to get a PRE style outside of
   <pre> </pre>.*)
let escape s =
  let escape_strings =
    if !pre then
      base_escape_strings
    else
      base_escape_strings @ prelike_escape_strings in
  List.fold_left
    (fun acc -> fun (s, s2) -> Str.global_replace (Str.regexp s) s2 acc)
    s
    escape_strings

(** Escape the strings which would clash with html syntax. *)
let escape_base s =
  List.fold_left
    (fun acc -> fun (s, s2) -> Str.global_replace (Str.regexp s) s2 acc)
    s
    base_escape_strings

(** The output functions *)

let print ?(esc=true) s =
  Format.pp_print_string !fmt (if esc then escape s else s)

let print_class ?(esc=true) cl s =
  print ~esc: false ("<span class=\""^cl^"\">"^
                     (if esc then escape s else s)^
                     "</span>")

(** The table of keywords with colors *)
let create_hashtable size init =
  let tbl = Hashtbl.create size in
  List.iter (fun (key, data) -> Hashtbl.add tbl key data) init;
  tbl

(** The function used to return html code for the given comment body. *)
let html_of_comment = ref
    (fun (_ : string) -> "<b>Odoc_ocamlhtml.html_of_comment not initialized</b>")

let keyword_table =
  create_hashtable 149 [
    "and", "keyword" ;
    "as", "keyword" ;
    "assert", "keyword" ;
    "begin", "keyword" ;
    "class", "keyword" ;
    "constraint", "keyword" ;
    "do", "keyword" ;
    "done", "keyword" ;
    "downto", "keyword" ;
    "else", "keyword" ;
    "end", "keyword" ;
    "exception", "keyword" ;
    "external", "keyword" ;
    "false", "keyword" ;
    "for", "keyword" ;
    "fun", "keyword" ;
    "function", "keyword" ;
    "functor", "keyword" ;
    "if", "keyword" ;
    "in", "keyword" ;
    "include", "keyword" ;
    "inherit", "keyword" ;
    "initializer", "keyword" ;
    "lazy", "keyword" ;
    "let", "keyword" ;
    "match", "keyword" ;
    "method", "keyword" ;
    "module", "keyword" ;
    "mutable", "keyword" ;
    "new", "keyword" ;
    "object", "keyword" ;
    "of", "keyword" ;
    "open", "keyword" ;
    "or", "keyword" ;
    "parser", "keyword" ;
    "private", "keyword" ;
    "rec", "keyword" ;
    "sig", "keyword" ;
    "struct", "keyword" ;
    "then", "keyword" ;
    "to", "keyword" ;
    "true", "keyword" ;
    "try", "keyword" ;
    "type", "keyword" ;
    "val", "keyword" ;
    "virtual", "keyword" ;
    "when", "keyword" ;
    "while", "keyword" ;
    "with", "keyword" ;

    "mod", "keyword" ;
    "land", "keyword" ;
    "lor", "keyword" ;
    "lxor", "keyword" ;
    "lsl", "keyword" ;
    "lsr", "keyword" ;
    "asr", "keyword" ;
]

let kwsign_class = "keywordsign"
let constructor_class = "constructor"
let comment_class = "comment"
let string_class = "string"
let code_class = "code"


(** To buffer and print comments *)


let margin = ref 0

let comment_buffer = Buffer.create 32
let reset_comment_buffer () = Buffer.reset comment_buffer
let store_comment_char = Buffer.add_char comment_buffer

let make_margin () =
  let rec iter n =
    if n <= 0 then ""
    else "&nbsp;"^(iter (n-1))
  in
  iter !margin

let print_comment () =
  let s = Buffer.contents comment_buffer in
  let len = String.length s in
  let code =
    if len < 1 then
      "<span class=\""^comment_class^"\">(*"^(escape s)^"*)</span>"
    else
      match s.[0] with
        '*' ->
          (
           try
             let html = !html_of_comment (String.sub s 1 (len-1)) in
             "</code><table><tr><td>"^(make_margin ())^"</td><td>"^
             "<span class=\""^comment_class^"\">"^
             "(**"^html^"*)"^
             "</span></td></tr></table><code class=\""^code_class^"\">"
           with
             e ->
               prerr_endline (Printexc.to_string e);
               "<span class=\""^comment_class^"\">(*"^(escape s)^"*)</span>"
          )
      | _ ->
          "<span class=\""^comment_class^"\">(*"^(escape s)^"*)</span>"
  in
  print ~esc: false code

(** To buffer string literals *)

let string_buffer = Buffer.create 32
let reset_string_buffer () = Buffer.reset string_buffer
let store_string_char = Buffer.add_char string_buffer
let get_stored_string () =
  Buffer.contents string_buffer

(** To store the position of the beginning of a string and comment *)
let string_start_pos = ref 0
let comment_start_pos = ref []

(** Normalizing utf-8 *)
let normalize raw_name =
  (* we are printing documentation, it is too late to be strict *)
  match Misc.Utf8_lexeme.normalize raw_name with
  | Error s -> s
  | Ok name -> name


# 218 "ocamldoc/odoc_ocamlhtml.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\191\255\192\255\224\000\003\001\038\001\073\001\108\001\
    \205\255\143\001\180\001\032\000\213\255\067\000\217\001\252\001\
    \069\000\071\000\084\000\031\002\230\255\232\255\235\255\066\002\
    \122\000\101\002\092\000\219\000\246\255\089\000\120\002\193\002\
    \154\003\121\004\088\005\049\006\255\255\049\007\144\007\050\008\
    \012\009\252\255\012\010\107\010\013\011\231\011\250\255\231\012\
    \112\000\224\000\099\000\132\000\229\003\200\004\130\002\101\000\
    \252\003\006\004\144\007\097\000\245\255\028\004\112\000\244\255\
    \121\004\113\000\243\255\112\000\241\255\070\013\240\255\203\005\
    \255\004\001\000\239\255\007\000\108\013\143\013\178\013\213\013\
    \226\255\222\255\223\255\224\255\220\255\248\013\214\255\215\255\
    \211\255\208\255\027\014\204\255\206\255\062\014\097\014\249\000\
    \252\255\253\255\115\000\117\000\255\255\254\255\020\001\249\255\
    \250\255\175\007\255\255\050\008\223\004\253\255\047\001\038\001\
    \088\005\252\255\107\010\251\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\062\000\059\000\058\000\053\000\056\000\
    \255\255\048\000\045\000\043\000\255\255\039\000\038\000\036\000\
    \034\000\030\000\028\000\054\000\255\255\255\255\255\255\018\000\
    \017\000\024\000\022\000\021\000\255\255\007\000\007\000\006\000\
    \006\000\004\000\002\000\001\000\255\255\006\000\057\000\255\255\
    \255\255\255\255\255\255\026\000\255\255\255\255\255\255\255\255\
    \255\255\008\000\255\255\255\255\255\255\007\000\007\000\007\000\
    \008\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\013\000\255\255\061\000\255\255\255\255\
    \255\255\016\000\255\255\255\255\019\000\060\000\055\000\027\000\
    \255\255\255\255\255\255\255\255\255\255\037\000\255\255\255\255\
    \255\255\255\255\046\000\255\255\255\255\056\000\052\000\255\255\
    \255\255\255\255\003\000\003\000\255\255\255\255\255\255\255\255\
    \255\255\006\000\255\255\255\255\255\255\255\255\001\000\001\000\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\059\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\037\000\255\255\040\000\
    \255\255\000\000\042\000\255\255\045\000\255\255\000\000\047\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\000\000\
    \255\255\255\255\000\000\255\255\000\000\255\255\000\000\075\000\
    \255\255\255\255\000\000\075\000\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\255\255\255\255\096\000\
    \000\000\000\000\255\255\255\255\000\000\000\000\103\000\000\000\
    \000\000\255\255\000\000\255\255\255\255\000\000\255\255\255\255\
    \255\255\000\000\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\036\000\036\000\074\000\036\000\036\000\000\000\000\000\
    \000\000\074\000\000\000\000\000\073\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \036\000\007\000\028\000\024\000\005\000\003\000\023\000\027\000\
    \026\000\021\000\025\000\006\000\020\000\019\000\018\000\003\000\
    \030\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\017\000\016\000\015\000\014\000\009\000\033\000\
    \004\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\013\000\088\000\012\000\004\000\035\000\
    \022\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\011\000\010\000\008\000\034\000\086\000\
    \084\000\083\000\080\000\072\000\082\000\081\000\067\000\049\000\
    \060\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\055\000\055\000\055\000\055\000\063\000\
    \066\000\068\000\072\000\057\000\101\000\057\000\048\000\100\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\000\000\000\000\048\000\087\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \002\000\003\000\255\255\000\000\003\000\003\000\003\000\255\255\
    \000\000\000\000\003\000\003\000\000\000\003\000\003\000\003\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\099\000\000\000\098\000\004\000\048\000\000\000\004\000\
    \004\000\004\000\000\000\000\000\000\000\004\000\004\000\111\000\
    \004\000\004\000\004\000\000\000\000\000\000\000\106\000\058\000\
    \111\000\111\000\000\000\000\000\000\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\000\000\048\000\111\000\005\000\
    \000\000\000\000\005\000\005\000\005\000\000\000\000\000\111\000\
    \005\000\005\000\000\000\005\000\005\000\005\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\000\000\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\000\000\
    \000\000\000\000\077\000\000\000\000\000\077\000\077\000\077\000\
    \105\000\000\000\000\000\077\000\077\000\000\000\077\000\077\000\
    \077\000\000\000\000\000\000\000\000\000\000\000\000\000\004\000\
    \000\000\004\000\000\000\077\000\005\000\077\000\077\000\077\000\
    \077\000\077\000\000\000\000\000\000\000\093\000\000\000\000\000\
    \093\000\093\000\093\000\000\000\000\000\000\000\093\000\093\000\
    \000\000\093\000\093\000\093\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\005\000\000\000\005\000\000\000\093\000\077\000\
    \093\000\094\000\093\000\093\000\093\000\000\000\000\000\000\000\
    \005\000\000\000\000\000\005\000\005\000\005\000\000\000\000\000\
    \000\000\005\000\005\000\000\000\005\000\005\000\005\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\077\000\000\000\077\000\
    \000\000\005\000\093\000\005\000\005\000\005\000\005\000\005\000\
    \000\000\000\000\000\000\000\000\000\000\005\000\000\000\000\000\
    \005\000\005\000\005\000\255\255\000\000\000\000\005\000\005\000\
    \000\000\005\000\005\000\005\000\000\000\000\000\000\000\000\000\
    \093\000\000\000\093\000\000\000\092\000\005\000\005\000\000\000\
    \005\000\005\000\005\000\005\000\005\000\000\000\000\000\000\000\
    \000\000\097\000\005\000\000\000\000\000\005\000\005\000\005\000\
    \000\000\000\000\000\000\005\000\005\000\000\000\005\000\005\000\
    \005\000\000\000\000\000\005\000\091\000\005\000\000\000\000\000\
    \000\000\089\000\005\000\005\000\104\000\005\000\005\000\005\000\
    \005\000\005\000\000\000\000\000\000\000\005\000\000\000\000\000\
    \005\000\005\000\005\000\000\000\000\000\000\000\005\000\005\000\
    \000\000\085\000\005\000\005\000\000\000\000\000\000\000\000\000\
    \090\000\000\000\005\000\000\000\000\000\000\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\000\000\000\000\000\000\
    \077\000\000\000\000\000\077\000\077\000\077\000\000\000\000\000\
    \000\000\077\000\077\000\000\000\077\000\078\000\077\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\005\000\000\000\005\000\
    \000\000\077\000\005\000\077\000\077\000\079\000\077\000\077\000\
    \000\000\000\000\000\000\005\000\000\000\000\000\005\000\005\000\
    \076\000\000\000\000\000\000\000\005\000\005\000\000\000\005\000\
    \005\000\005\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \005\000\000\000\005\000\000\000\005\000\077\000\005\000\005\000\
    \005\000\005\000\005\000\000\000\000\000\000\000\003\000\000\000\
    \000\000\003\000\003\000\003\000\000\000\000\000\070\000\069\000\
    \003\000\000\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\077\000\000\000\077\000\000\000\003\000\
    \005\000\003\000\003\000\003\000\003\000\003\000\049\000\000\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\050\000\000\000\000\000\048\000\005\000\000\000\
    \005\000\000\000\000\000\003\000\000\000\000\000\000\000\051\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \052\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\050\000\000\000\000\000\048\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\000\000\000\000\000\000\051\000\
    \032\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \052\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\000\000\000\000\000\000\
    \032\000\000\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\000\000\000\000\000\000\
    \000\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\032\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\000\000\000\000\000\000\
    \000\000\032\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\000\000\000\000\
    \000\000\000\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\038\000\000\000\000\000\038\000\038\000\038\000\
    \000\000\000\000\000\000\038\000\038\000\000\000\038\000\038\000\
    \038\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\038\000\000\000\038\000\038\000\038\000\
    \043\000\038\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\000\000\000\000\000\000\038\000\
    \045\000\000\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\000\000\038\000\000\000\038\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \072\000\053\000\053\000\053\000\053\000\053\000\053\000\112\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\000\000\000\000\000\000\000\000\000\000\000\000\072\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\053\000\053\000\053\000\053\000\053\000\053\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\038\000\000\000\000\000\038\000\038\000\038\000\000\000\
    \000\000\000\000\038\000\038\000\000\000\038\000\038\000\038\000\
    \113\000\113\000\113\000\113\000\113\000\113\000\113\000\113\000\
    \113\000\113\000\038\000\000\000\038\000\038\000\038\000\038\000\
    \038\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\000\000\000\000\000\000\038\000\040\000\
    \000\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\000\000\038\000\074\000\038\000\000\000\
    \073\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \032\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\000\000\000\000\000\000\
    \032\000\000\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \032\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\255\255\255\255\255\255\255\255\
    \032\000\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\255\255\255\255\255\255\255\255\
    \255\255\038\000\000\000\000\000\038\000\038\000\038\000\062\000\
    \000\000\111\000\038\000\038\000\110\000\038\000\038\000\038\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\038\000\000\000\038\000\038\000\038\000\038\000\
    \038\000\109\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\000\000\000\000\000\000\062\000\000\000\038\000\000\000\
    \000\000\000\000\062\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\062\000\000\000\
    \000\000\000\000\062\000\000\000\062\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\109\000\038\000\000\000\038\000\000\000\
    \000\000\109\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\109\000\000\000\000\000\
    \000\000\109\000\000\000\109\000\000\000\000\000\000\000\107\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\041\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\255\255\255\255\255\255\
    \255\255\000\000\255\255\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\255\255\255\255\255\255\
    \255\255\255\255\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\255\255\040\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\041\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\000\000\
    \000\000\000\000\000\000\040\000\000\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\040\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\041\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\255\255\
    \255\255\255\255\255\255\040\000\255\255\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\255\255\
    \255\255\255\255\255\255\255\255\038\000\000\000\000\000\038\000\
    \038\000\038\000\000\000\000\000\000\000\038\000\038\000\000\000\
    \038\000\038\000\038\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\038\000\000\000\038\000\
    \038\000\038\000\038\000\038\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\000\000\000\000\
    \000\000\038\000\000\000\000\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\000\000\038\000\
    \000\000\038\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\046\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\255\255\045\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\046\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\000\000\000\000\000\000\000\000\045\000\000\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\045\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\046\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\255\255\255\255\255\255\255\255\045\000\255\255\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\255\255\255\255\255\255\255\255\255\255\069\000\
    \000\000\000\000\069\000\069\000\069\000\000\000\000\000\000\000\
    \069\000\069\000\000\000\069\000\069\000\069\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \069\000\000\000\069\000\069\000\069\000\069\000\069\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\005\000\000\000\000\000\
    \005\000\005\000\005\000\000\000\000\000\000\000\005\000\005\000\
    \000\000\005\000\005\000\005\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\069\000\000\000\005\000\000\000\
    \005\000\005\000\005\000\005\000\005\000\000\000\000\000\000\000\
    \077\000\000\000\000\000\077\000\077\000\077\000\000\000\000\000\
    \000\000\077\000\077\000\000\000\077\000\077\000\077\000\000\000\
    \000\000\000\000\069\000\000\000\069\000\000\000\000\000\000\000\
    \000\000\077\000\005\000\077\000\077\000\077\000\077\000\077\000\
    \000\000\000\000\000\000\077\000\000\000\000\000\077\000\077\000\
    \077\000\000\000\000\000\000\000\077\000\077\000\000\000\077\000\
    \077\000\077\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \005\000\000\000\005\000\000\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\000\000\000\000\000\000\077\000\000\000\
    \000\000\077\000\077\000\077\000\000\000\000\000\000\000\077\000\
    \077\000\000\000\077\000\077\000\077\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\077\000\000\000\077\000\000\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\000\000\000\000\
    \000\000\005\000\000\000\000\000\005\000\005\000\005\000\000\000\
    \000\000\000\000\005\000\005\000\000\000\005\000\005\000\005\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\077\000\000\000\
    \077\000\000\000\005\000\077\000\005\000\005\000\005\000\005\000\
    \005\000\000\000\000\000\000\000\005\000\000\000\000\000\005\000\
    \005\000\005\000\000\000\000\000\000\000\005\000\005\000\000\000\
    \005\000\005\000\005\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\077\000\000\000\077\000\000\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\000\000\000\000\000\000\093\000\
    \000\000\000\000\093\000\093\000\093\000\000\000\000\000\000\000\
    \093\000\093\000\000\000\093\000\093\000\093\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\005\000\000\000\005\000\000\000\
    \093\000\005\000\093\000\093\000\093\000\093\000\093\000\000\000\
    \000\000\000\000\093\000\000\000\000\000\093\000\093\000\093\000\
    \000\000\000\000\000\000\093\000\093\000\000\000\093\000\093\000\
    \093\000\000\000\000\000\000\000\000\000\000\000\000\000\005\000\
    \000\000\005\000\000\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\093\000\000\000\093\000\000\000\000\000\093\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\093\000\000\000\093\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\073\000\000\000\000\000\255\255\255\255\
    \255\255\075\000\255\255\255\255\075\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\013\000\
    \016\000\017\000\018\000\024\000\017\000\017\000\026\000\029\000\
    \059\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\050\000\050\000\055\000\055\000\062\000\
    \065\000\067\000\024\000\048\000\098\000\048\000\029\000\099\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\255\255\255\255\029\000\013\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\027\000\255\255\003\000\003\000\003\000\075\000\
    \255\255\255\255\003\000\003\000\255\255\003\000\003\000\003\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\003\000\255\255\003\000\003\000\003\000\003\000\
    \003\000\095\000\255\255\095\000\004\000\049\000\255\255\004\000\
    \004\000\004\000\255\255\255\255\255\255\004\000\004\000\111\000\
    \004\000\004\000\004\000\255\255\255\255\255\255\102\000\027\000\
    \110\000\110\000\255\255\255\255\255\255\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\255\255\049\000\111\000\005\000\
    \255\255\255\255\005\000\005\000\005\000\255\255\255\255\110\000\
    \005\000\005\000\255\255\005\000\005\000\005\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\003\000\255\255\003\000\255\255\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\255\255\
    \255\255\255\255\006\000\255\255\255\255\006\000\006\000\006\000\
    \102\000\255\255\255\255\006\000\006\000\255\255\006\000\006\000\
    \006\000\255\255\255\255\255\255\255\255\255\255\255\255\004\000\
    \255\255\004\000\255\255\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\255\255\255\255\255\255\007\000\255\255\255\255\
    \007\000\007\000\007\000\255\255\255\255\255\255\007\000\007\000\
    \255\255\007\000\007\000\007\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\005\000\255\255\005\000\255\255\007\000\006\000\
    \007\000\007\000\007\000\007\000\007\000\255\255\255\255\255\255\
    \009\000\255\255\255\255\009\000\009\000\009\000\255\255\255\255\
    \255\255\009\000\009\000\255\255\009\000\009\000\009\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\006\000\255\255\006\000\
    \255\255\009\000\007\000\009\000\009\000\009\000\009\000\009\000\
    \255\255\255\255\255\255\255\255\255\255\010\000\255\255\255\255\
    \010\000\010\000\010\000\027\000\255\255\255\255\010\000\010\000\
    \255\255\010\000\010\000\010\000\255\255\255\255\255\255\255\255\
    \007\000\255\255\007\000\255\255\009\000\009\000\010\000\255\255\
    \010\000\010\000\010\000\010\000\010\000\255\255\255\255\255\255\
    \255\255\095\000\014\000\255\255\255\255\014\000\014\000\014\000\
    \255\255\255\255\255\255\014\000\014\000\255\255\014\000\014\000\
    \014\000\255\255\255\255\009\000\009\000\009\000\255\255\255\255\
    \255\255\010\000\010\000\014\000\102\000\014\000\014\000\014\000\
    \014\000\014\000\255\255\255\255\255\255\015\000\255\255\255\255\
    \015\000\015\000\015\000\255\255\255\255\255\255\015\000\015\000\
    \255\255\015\000\015\000\015\000\255\255\255\255\255\255\255\255\
    \010\000\255\255\010\000\255\255\255\255\255\255\015\000\014\000\
    \015\000\015\000\015\000\015\000\015\000\255\255\255\255\255\255\
    \019\000\255\255\255\255\019\000\019\000\019\000\255\255\255\255\
    \255\255\019\000\019\000\255\255\019\000\019\000\019\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\014\000\255\255\014\000\
    \255\255\019\000\015\000\019\000\019\000\019\000\019\000\019\000\
    \255\255\255\255\255\255\023\000\255\255\255\255\023\000\023\000\
    \023\000\255\255\255\255\255\255\023\000\023\000\255\255\023\000\
    \023\000\023\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \015\000\255\255\015\000\255\255\023\000\019\000\023\000\023\000\
    \023\000\023\000\023\000\255\255\255\255\255\255\025\000\255\255\
    \255\255\025\000\025\000\025\000\255\255\255\255\025\000\025\000\
    \025\000\255\255\025\000\025\000\025\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\019\000\255\255\019\000\255\255\025\000\
    \023\000\025\000\025\000\025\000\025\000\025\000\030\000\255\255\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\030\000\255\255\255\255\030\000\023\000\255\255\
    \023\000\255\255\255\255\025\000\255\255\255\255\255\255\030\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \030\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\030\000\255\255\255\255\030\000\255\255\255\255\
    \255\255\025\000\255\255\025\000\255\255\255\255\255\255\030\000\
    \031\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \030\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\255\255\255\255\255\255\255\255\
    \031\000\255\255\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\255\255\255\255\255\255\255\255\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\032\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\255\255\255\255\255\255\
    \255\255\032\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\052\000\052\000\
    \052\000\052\000\052\000\052\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\052\000\052\000\
    \052\000\052\000\052\000\052\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\255\255\255\255\
    \255\255\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\033\000\255\255\255\255\033\000\033\000\033\000\
    \255\255\255\255\255\255\033\000\033\000\255\255\033\000\033\000\
    \033\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\033\000\255\255\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\255\255\255\255\255\255\033\000\
    \033\000\255\255\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\255\255\033\000\255\255\033\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \072\000\053\000\053\000\053\000\053\000\053\000\053\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\255\255\255\255\255\255\255\255\255\255\255\255\072\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\053\000\053\000\053\000\053\000\053\000\053\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\034\000\255\255\255\255\034\000\034\000\034\000\255\255\
    \255\255\255\255\034\000\034\000\255\255\034\000\034\000\034\000\
    \112\000\112\000\112\000\112\000\112\000\112\000\112\000\112\000\
    \112\000\112\000\034\000\255\255\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\255\255\255\255\255\255\034\000\034\000\
    \255\255\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\255\255\034\000\071\000\034\000\255\255\
    \071\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \035\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\255\255\255\255\255\255\
    \035\000\255\255\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\071\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\038\000\255\255\255\255\038\000\038\000\038\000\058\000\
    \255\255\105\000\038\000\038\000\105\000\038\000\038\000\038\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\038\000\255\255\038\000\038\000\038\000\038\000\
    \038\000\105\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\105\000\
    \105\000\105\000\105\000\105\000\105\000\105\000\105\000\105\000\
    \105\000\255\255\255\255\255\255\058\000\255\255\038\000\255\255\
    \255\255\255\255\058\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\058\000\255\255\
    \255\255\255\255\058\000\255\255\058\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\105\000\038\000\255\255\038\000\255\255\
    \255\255\105\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\105\000\255\255\255\255\
    \255\255\105\000\255\255\105\000\255\255\255\255\255\255\105\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\037\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\255\255\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\039\000\039\000\039\000\
    \039\000\255\255\039\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\040\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\255\255\
    \255\255\255\255\255\255\040\000\255\255\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\043\000\255\255\255\255\043\000\
    \043\000\043\000\255\255\255\255\255\255\043\000\043\000\255\255\
    \043\000\043\000\043\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\043\000\255\255\043\000\
    \043\000\043\000\043\000\043\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\255\255\255\255\
    \255\255\043\000\255\255\255\255\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\255\255\043\000\
    \255\255\043\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\042\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\255\255\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \044\000\044\000\044\000\044\000\255\255\044\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\044\000\045\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\255\255\255\255\255\255\255\255\045\000\255\255\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\045\000\
    \045\000\045\000\045\000\045\000\045\000\045\000\045\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\069\000\
    \255\255\255\255\069\000\069\000\069\000\255\255\255\255\255\255\
    \069\000\069\000\255\255\069\000\069\000\069\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \069\000\255\255\069\000\069\000\069\000\069\000\069\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\076\000\255\255\255\255\
    \076\000\076\000\076\000\255\255\255\255\255\255\076\000\076\000\
    \255\255\076\000\076\000\076\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\069\000\255\255\076\000\255\255\
    \076\000\076\000\076\000\076\000\076\000\255\255\255\255\255\255\
    \077\000\255\255\255\255\077\000\077\000\077\000\255\255\255\255\
    \255\255\077\000\077\000\255\255\077\000\077\000\077\000\255\255\
    \255\255\255\255\069\000\255\255\069\000\255\255\255\255\255\255\
    \255\255\077\000\076\000\077\000\077\000\077\000\077\000\077\000\
    \255\255\255\255\255\255\078\000\255\255\255\255\078\000\078\000\
    \078\000\255\255\255\255\255\255\078\000\078\000\255\255\078\000\
    \078\000\078\000\255\255\255\255\255\255\255\255\255\255\047\000\
    \076\000\255\255\076\000\255\255\078\000\077\000\078\000\078\000\
    \078\000\078\000\078\000\255\255\255\255\255\255\079\000\255\255\
    \255\255\079\000\079\000\079\000\255\255\255\255\255\255\079\000\
    \079\000\255\255\079\000\079\000\079\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\077\000\255\255\077\000\255\255\079\000\
    \078\000\079\000\079\000\079\000\079\000\079\000\255\255\255\255\
    \255\255\085\000\255\255\255\255\085\000\085\000\085\000\255\255\
    \255\255\255\255\085\000\085\000\255\255\085\000\085\000\085\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\078\000\255\255\
    \078\000\255\255\085\000\079\000\085\000\085\000\085\000\085\000\
    \085\000\255\255\255\255\255\255\090\000\255\255\255\255\090\000\
    \090\000\090\000\255\255\255\255\255\255\090\000\090\000\255\255\
    \090\000\090\000\090\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\079\000\255\255\079\000\255\255\090\000\085\000\090\000\
    \090\000\090\000\090\000\090\000\255\255\255\255\255\255\093\000\
    \255\255\255\255\093\000\093\000\093\000\255\255\255\255\255\255\
    \093\000\093\000\255\255\093\000\093\000\093\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\085\000\255\255\085\000\255\255\
    \093\000\090\000\093\000\093\000\093\000\093\000\093\000\255\255\
    \255\255\255\255\094\000\255\255\255\255\094\000\094\000\094\000\
    \255\255\255\255\255\255\094\000\094\000\255\255\094\000\094\000\
    \094\000\255\255\255\255\255\255\255\255\255\255\255\255\090\000\
    \255\255\090\000\255\255\094\000\093\000\094\000\094\000\094\000\
    \094\000\094\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\093\000\255\255\093\000\255\255\255\255\094\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\094\000\255\255\094\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_base_code =
   "";
  Lexing.lex_backtrk_code =
   "";
  Lexing.lex_default_code =
   "";
  Lexing.lex_trans_code =
   "";
  Lexing.lex_check_code =
   "";
  Lexing.lex_code =
   "";
}

let rec token lexbuf =
   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 239 "ocamldoc/odoc_ocamlhtml.mll"
      (
        let s = Lexing.lexeme lexbuf in
        (
         match s with
           " " -> incr margin
         | "\t" -> margin := !margin + 8
         | "\n" -> margin := 0
         | _ -> ()
        );
        print s;
        token lexbuf
      )
# 1288 "ocamldoc/odoc_ocamlhtml.ml"

  | 1 ->
# 252 "ocamldoc/odoc_ocamlhtml.mll"
      ( print "_" ; token lexbuf )
# 1293 "ocamldoc/odoc_ocamlhtml.ml"

  | 2 ->
# 253 "ocamldoc/odoc_ocamlhtml.mll"
         ( print "~" ; token lexbuf )
# 1298 "ocamldoc/odoc_ocamlhtml.ml"

  | 3 ->
let
# 254 "ocamldoc/odoc_ocamlhtml.mll"
                      raw_id
# 1304 "ocamldoc/odoc_ocamlhtml.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 255 "ocamldoc/odoc_ocamlhtml.mll"
      ( let s = Lexing.lexeme lexbuf in
        let name = normalize raw_id in
        if Hashtbl.mem keyword_table name then
          raise (Error(Keyword_as_label name, Lexing.lexeme_start lexbuf,
                       Lexing.lexeme_end lexbuf));
        print s ; token lexbuf )
# 1313 "ocamldoc/odoc_ocamlhtml.ml"

  | 4 ->
# 261 "ocamldoc/odoc_ocamlhtml.mll"
         ( print "?" ; token lexbuf )
# 1318 "ocamldoc/odoc_ocamlhtml.ml"

  | 5 ->
let
# 262 "ocamldoc/odoc_ocamlhtml.mll"
                      raw_id
# 1324 "ocamldoc/odoc_ocamlhtml.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 263 "ocamldoc/odoc_ocamlhtml.mll"
      (
        let name = normalize raw_id in
        if Hashtbl.mem keyword_table name then
          raise (Error(Keyword_as_label name, Lexing.lexeme_start lexbuf,
                       Lexing.lexeme_end lexbuf));
        print "?"; print name ; print ":"; token lexbuf )
# 1333 "ocamldoc/odoc_ocamlhtml.ml"

  | 6 ->
let
# 269 "ocamldoc/odoc_ocamlhtml.mll"
                  raw_id
# 1339 "ocamldoc/odoc_ocamlhtml.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 270 "ocamldoc/odoc_ocamlhtml.mll"
      (  let s = normalize raw_id in
         if Misc.Utf8_lexeme.is_capitalized s then
            (print_class constructor_class (Lexing.lexeme lexbuf);
            token lexbuf)
         else try
            let cl = Hashtbl.find keyword_table s in
            (print_class cl s ; token lexbuf )
          with Not_found ->
            (print s ; token lexbuf ))
# 1351 "ocamldoc/odoc_ocamlhtml.ml"

  | 7 ->
# 280 "ocamldoc/odoc_ocamlhtml.mll"
      ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1356 "ocamldoc/odoc_ocamlhtml.ml"

  | 8 ->
# 282 "ocamldoc/odoc_ocamlhtml.mll"
      ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1361 "ocamldoc/odoc_ocamlhtml.ml"

  | 9 ->
# 284 "ocamldoc/odoc_ocamlhtml.mll"
      ( reset_string_buffer();
        let string_start = Lexing.lexeme_start lexbuf in
        string_start_pos := string_start;
        string lexbuf;
        lexbuf.Lexing.lex_start_pos <-
          string_start - lexbuf.Lexing.lex_abs_pos;
        print_class string_class ("\""^(get_stored_string())^"\"") ;
        token lexbuf )
# 1373 "ocamldoc/odoc_ocamlhtml.ml"

  | 10 ->
# 293 "ocamldoc/odoc_ocamlhtml.mll"
      ( print_class string_class (Lexing.lexeme lexbuf) ;
        token lexbuf )
# 1379 "ocamldoc/odoc_ocamlhtml.ml"

  | 11 ->
# 296 "ocamldoc/odoc_ocamlhtml.mll"
      ( print_class string_class (Lexing.lexeme lexbuf ) ;
        token lexbuf )
# 1385 "ocamldoc/odoc_ocamlhtml.ml"

  | 12 ->
# 299 "ocamldoc/odoc_ocamlhtml.mll"
      ( print_class string_class (Lexing.lexeme lexbuf ) ;
        token lexbuf )
# 1391 "ocamldoc/odoc_ocamlhtml.ml"

  | 13 ->
# 302 "ocamldoc/odoc_ocamlhtml.mll"
      (
        reset_comment_buffer ();
        comment_start_pos := [Lexing.lexeme_start lexbuf];
        comment lexbuf ;
        print_comment ();
        token lexbuf )
# 1401 "ocamldoc/odoc_ocamlhtml.ml"

  | 14 ->
# 309 "ocamldoc/odoc_ocamlhtml.mll"
      ( reset_comment_buffer ();
        comment_start_pos := [Lexing.lexeme_start lexbuf];
        comment lexbuf ;
        print_comment ();
        token lexbuf
      )
# 1411 "ocamldoc/odoc_ocamlhtml.ml"

  | 15 ->
# 316 "ocamldoc/odoc_ocamlhtml.mll"
      ( lexbuf.Lexing.lex_curr_pos <- lexbuf.Lexing.lex_curr_pos - 1;
        lexbuf.Lexing.lex_curr_p <-
          { lexbuf.Lexing.lex_curr_p with
            pos_cnum = lexbuf.Lexing.lex_curr_p.pos_cnum - 1
          } ;
        print (Lexing.lexeme lexbuf) ;
        token lexbuf
      )
# 1423 "ocamldoc/odoc_ocamlhtml.ml"

  | 16 ->
# 326 "ocamldoc/odoc_ocamlhtml.mll"
      (
        print (Lexing.lexeme lexbuf);
        token lexbuf
      )
# 1431 "ocamldoc/odoc_ocamlhtml.ml"

  | 17 ->
# 330 "ocamldoc/odoc_ocamlhtml.mll"
         ( print_class kwsign_class (Lexing.lexeme lexbuf) ; token lexbuf )
# 1436 "ocamldoc/odoc_ocamlhtml.ml"

  | 18 ->
# 331 "ocamldoc/odoc_ocamlhtml.mll"
         ( print_class kwsign_class (Lexing.lexeme lexbuf) ; token lexbuf )
# 1441 "ocamldoc/odoc_ocamlhtml.ml"

  | 19 ->
# 332 "ocamldoc/odoc_ocamlhtml.mll"
         ( print_class kwsign_class (Lexing.lexeme lexbuf) ; token lexbuf )
# 1446 "ocamldoc/odoc_ocamlhtml.ml"

  | 20 ->
# 333 "ocamldoc/odoc_ocamlhtml.mll"
         ( print_class kwsign_class (Lexing.lexeme lexbuf) ; token lexbuf )
# 1451 "ocamldoc/odoc_ocamlhtml.ml"

  | 21 ->
# 334 "ocamldoc/odoc_ocamlhtml.mll"
         ( print_class kwsign_class (Lexing.lexeme lexbuf) ; token lexbuf )
# 1456 "ocamldoc/odoc_ocamlhtml.ml"

  | 22 ->
# 335 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1461 "ocamldoc/odoc_ocamlhtml.ml"

  | 23 ->
# 336 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1466 "ocamldoc/odoc_ocamlhtml.ml"

  | 24 ->
# 337 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1471 "ocamldoc/odoc_ocamlhtml.ml"

  | 25 ->
# 338 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1476 "ocamldoc/odoc_ocamlhtml.ml"

  | 26 ->
# 339 "ocamldoc/odoc_ocamlhtml.mll"
         ( print_class kwsign_class (Lexing.lexeme lexbuf) ; token lexbuf )
# 1481 "ocamldoc/odoc_ocamlhtml.ml"

  | 27 ->
# 340 "ocamldoc/odoc_ocamlhtml.mll"
         ( print_class kwsign_class (Lexing.lexeme lexbuf) ; token lexbuf )
# 1486 "ocamldoc/odoc_ocamlhtml.ml"

  | 28 ->
# 341 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1491 "ocamldoc/odoc_ocamlhtml.ml"

  | 29 ->
# 342 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1496 "ocamldoc/odoc_ocamlhtml.ml"

  | 30 ->
# 343 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1501 "ocamldoc/odoc_ocamlhtml.ml"

  | 31 ->
# 344 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1506 "ocamldoc/odoc_ocamlhtml.ml"

  | 32 ->
# 345 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1511 "ocamldoc/odoc_ocamlhtml.ml"

  | 33 ->
# 346 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1516 "ocamldoc/odoc_ocamlhtml.ml"

  | 34 ->
# 347 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1521 "ocamldoc/odoc_ocamlhtml.ml"

  | 35 ->
# 348 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1526 "ocamldoc/odoc_ocamlhtml.ml"

  | 36 ->
# 349 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1531 "ocamldoc/odoc_ocamlhtml.ml"

  | 37 ->
# 350 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1536 "ocamldoc/odoc_ocamlhtml.ml"

  | 38 ->
# 351 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1541 "ocamldoc/odoc_ocamlhtml.ml"

  | 39 ->
# 352 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1546 "ocamldoc/odoc_ocamlhtml.ml"

  | 40 ->
# 353 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1551 "ocamldoc/odoc_ocamlhtml.ml"

  | 41 ->
# 354 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1556 "ocamldoc/odoc_ocamlhtml.ml"

  | 42 ->
# 355 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1561 "ocamldoc/odoc_ocamlhtml.ml"

  | 43 ->
# 356 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1566 "ocamldoc/odoc_ocamlhtml.ml"

  | 44 ->
# 357 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1571 "ocamldoc/odoc_ocamlhtml.ml"

  | 45 ->
# 358 "ocamldoc/odoc_ocamlhtml.mll"
         ( print_class kwsign_class (Lexing.lexeme lexbuf) ; token lexbuf )
# 1576 "ocamldoc/odoc_ocamlhtml.ml"

  | 46 ->
# 359 "ocamldoc/odoc_ocamlhtml.mll"
         ( print_class kwsign_class (Lexing.lexeme lexbuf) ; token lexbuf )
# 1581 "ocamldoc/odoc_ocamlhtml.ml"

  | 47 ->
# 360 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1586 "ocamldoc/odoc_ocamlhtml.ml"

  | 48 ->
# 361 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1591 "ocamldoc/odoc_ocamlhtml.ml"

  | 49 ->
# 362 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1596 "ocamldoc/odoc_ocamlhtml.ml"

  | 50 ->
# 363 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1601 "ocamldoc/odoc_ocamlhtml.ml"

  | 51 ->
# 364 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1606 "ocamldoc/odoc_ocamlhtml.ml"

  | 52 ->
# 366 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1611 "ocamldoc/odoc_ocamlhtml.ml"

  | 53 ->
# 367 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1616 "ocamldoc/odoc_ocamlhtml.ml"

  | 54 ->
# 368 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1621 "ocamldoc/odoc_ocamlhtml.ml"

  | 55 ->
# 369 "ocamldoc/odoc_ocamlhtml.mll"
         ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1626 "ocamldoc/odoc_ocamlhtml.ml"

  | 56 ->
# 372 "ocamldoc/odoc_ocamlhtml.mll"
            ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1631 "ocamldoc/odoc_ocamlhtml.ml"

  | 57 ->
# 374 "ocamldoc/odoc_ocamlhtml.mll"
            ( print_class kwsign_class (Lexing.lexeme lexbuf) ; token lexbuf )
# 1636 "ocamldoc/odoc_ocamlhtml.ml"

  | 58 ->
# 376 "ocamldoc/odoc_ocamlhtml.mll"
            ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1641 "ocamldoc/odoc_ocamlhtml.ml"

  | 59 ->
# 378 "ocamldoc/odoc_ocamlhtml.mll"
            ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1646 "ocamldoc/odoc_ocamlhtml.ml"

  | 60 ->
# 380 "ocamldoc/odoc_ocamlhtml.mll"
            ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1651 "ocamldoc/odoc_ocamlhtml.ml"

  | 61 ->
# 382 "ocamldoc/odoc_ocamlhtml.mll"
            ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1656 "ocamldoc/odoc_ocamlhtml.ml"

  | 62 ->
# 384 "ocamldoc/odoc_ocamlhtml.mll"
            ( print (Lexing.lexeme lexbuf) ; token lexbuf )
# 1661 "ocamldoc/odoc_ocamlhtml.ml"

  | 63 ->
# 385 "ocamldoc/odoc_ocamlhtml.mll"
        ( () )
# 1666 "ocamldoc/odoc_ocamlhtml.ml"

  | 64 ->
# 387 "ocamldoc/odoc_ocamlhtml.mll"
      ( raise (Error(Illegal_character ((Lexing.lexeme lexbuf).[0]),
                     Lexing.lexeme_start lexbuf, Lexing.lexeme_end lexbuf)) )
# 1672 "ocamldoc/odoc_ocamlhtml.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and comment lexbuf =
   __ocaml_lex_comment_rec lexbuf 95
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 392 "ocamldoc/odoc_ocamlhtml.mll"
      ( comment_start_pos := Lexing.lexeme_start lexbuf :: !comment_start_pos;
        store_comment_char '(';
        store_comment_char '*';
        comment lexbuf;
      )
# 1688 "ocamldoc/odoc_ocamlhtml.ml"

  | 1 ->
# 398 "ocamldoc/odoc_ocamlhtml.mll"
      ( match !comment_start_pos with
        | [] -> assert false
        | [_] -> comment_start_pos := []
        | _ :: l ->
            store_comment_char '*';
            store_comment_char ')';
            comment_start_pos := l;
            comment lexbuf;
       )
# 1701 "ocamldoc/odoc_ocamlhtml.ml"

  | 2 ->
# 447 "ocamldoc/odoc_ocamlhtml.mll"
      ( let st = List.hd !comment_start_pos in
        raise (Error (Unterminated_comment, st, st + 2));
      )
# 1708 "ocamldoc/odoc_ocamlhtml.ml"

  | 3 ->
# 451 "ocamldoc/odoc_ocamlhtml.mll"
      ( store_comment_char(Lexing.lexeme_char lexbuf 0);
        comment lexbuf )
# 1714 "ocamldoc/odoc_ocamlhtml.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and string lexbuf =
   __ocaml_lex_string_rec lexbuf 102
and __ocaml_lex_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 456 "ocamldoc/odoc_ocamlhtml.mll"
      ( () )
# 1726 "ocamldoc/odoc_ocamlhtml.ml"

  | 1 ->
# 458 "ocamldoc/odoc_ocamlhtml.mll"
      ( string lexbuf )
# 1731 "ocamldoc/odoc_ocamlhtml.ml"

  | 2 ->
# 460 "ocamldoc/odoc_ocamlhtml.mll"
      ( Buffer.add_string string_buffer (Lexing.lexeme lexbuf) ;
        string lexbuf )
# 1737 "ocamldoc/odoc_ocamlhtml.ml"

  | 3 ->
# 463 "ocamldoc/odoc_ocamlhtml.mll"
      (
        Buffer.add_string string_buffer (Lexing.lexeme lexbuf) ;
        string lexbuf
      )
# 1745 "ocamldoc/odoc_ocamlhtml.ml"

  | 4 ->
# 468 "ocamldoc/odoc_ocamlhtml.mll"
      (  Buffer.add_string string_buffer (Lexing.lexeme lexbuf) ;
         string lexbuf )
# 1751 "ocamldoc/odoc_ocamlhtml.ml"

  | 5 ->
# 471 "ocamldoc/odoc_ocamlhtml.mll"
      ( raise (Error (Unterminated_string,
                      !string_start_pos, !string_start_pos+1)) )
# 1757 "ocamldoc/odoc_ocamlhtml.ml"

  | 6 ->
# 474 "ocamldoc/odoc_ocamlhtml.mll"
      ( store_string_char(Lexing.lexeme_char lexbuf 0);
        string lexbuf )
# 1763 "ocamldoc/odoc_ocamlhtml.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_string_rec lexbuf __ocaml_lex_state

;;

# 476 "ocamldoc/odoc_ocamlhtml.mll"
 

let html_of_code b ?(with_pre=true) code =
  let old_pre = !pre in
  let old_margin = !margin in
  let old_comment_buffer = Buffer.contents comment_buffer in
  let old_string_buffer = Buffer.contents string_buffer in
  let buf = Buffer.create 256 in
  let old_fmt = !fmt in
  fmt := Format.formatter_of_buffer buf ;
  pre := with_pre;
  margin := 0;

  let start = "<code class=\""^code_class^"\">" in
  let ending = "</code>" in
  let html =
    (
     try
       print ~esc: false start ;
       let lexbuf = Lexing.from_string code in
       token lexbuf;
       print ~esc: false ending ;
       Format.pp_print_flush !fmt () ;
       Buffer.contents buf
     with
       _ ->
         (* flush str_formatter because we already output
            something in it *)
         Format.pp_print_flush !fmt () ;
         start^code^ending
    )
  in
  pre := old_pre;
  margin := old_margin ;
  Buffer.reset comment_buffer;
  Buffer.add_string comment_buffer old_comment_buffer ;
  Buffer.reset string_buffer;
  Buffer.add_string string_buffer old_string_buffer ;
  fmt := old_fmt ;

  Buffer.add_string b html


# 1814 "ocamldoc/odoc_ocamlhtml.ml"
