# 1 "ocamldoc/odoc_text_lexer.mll"
 
(**************************************************************************)
(*                                                                        *)
(*                                 OCaml                                  *)
(*                                                                        *)
(*             Maxence Guesdon, projet Cristal, INRIA Rocquencourt        *)
(*                                                                        *)
(*   Copyright 2001 Institut National de Recherche en Informatique et     *)
(*     en Automatique.                                                    *)
(*                                                                        *)
(*   All rights reserved.  This file is distributed under the terms of    *)
(*   the GNU Lesser General Public License version 2.1, with the          *)
(*   special exception on linking described in the file LICENSE.          *)
(*                                                                        *)
(**************************************************************************)

(** The lexer for string to build text structures. *)

open Lexing
open Odoc_text_parser

let line_number = ref 0
let char_number = ref 0

(** this flag indicates whether we're in a string between begin_code and end_code tokens, to
   remember the number of open '[' and handle ']' correctly. *)
let open_brackets = ref 0

(** this flag indicates if we're in verbatim mode or not, to handle any special expression
   like a string when we're in verbatim mode.*)
let verb_mode = ref false

(** this flag indicates if we're in "target format" mode or not, to handle any special expression
   like a string when we're in this mode.*)
let target_mode = ref false

(** this flag indicates if we're in shortcut list mode or not, to handle end_shortcut_list correctly.*)
let shortcut_list_mode = ref false

(** this flag indicates if we're in an element reference. *)
let ele_ref_mode = ref false

(** this flag indicates if we're in a preformatted code string. *)
let code_pre_mode = ref false

let init () =
  open_brackets := 0;
  verb_mode := false;
  target_mode := false;
  shortcut_list_mode := false;
  ele_ref_mode := false ;
  code_pre_mode := false ;
  line_number := 0 ;
  char_number := 0

let incr_cpts lexbuf =
  let s = Lexing.lexeme lexbuf in
  let l = Str.split_delim (Str.regexp_string "\n") s in
  match List.rev l with
    [] -> () (* should not occur *)
  | [s2] -> (* no newline *)
      char_number := !char_number + (String.length s2)
  | s2 :: _ ->
      line_number := !line_number + ((List.length l) - 1) ;
      char_number := String.length s2


# 70 "ocamldoc/odoc_text_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\205\255\209\255\000\000\001\000\006\000\001\000\001\000\
    \002\000\083\000\000\000\254\255\019\000\255\255\001\000\001\000\
    \005\000\138\000\000\000\001\000\170\000\162\000\161\000\002\000\
    \006\000\007\000\232\000\002\000\003\000\242\000\006\000\007\000\
    \008\000\003\000\011\000\006\000\028\000\061\000\016\000\078\000\
    \018\000\235\255\001\001\253\255\252\255\251\255\108\000\132\000\
    \127\000\143\000\131\000\022\000\246\255\145\000\024\000\025\000\
    \243\255\026\000\242\255\033\000\241\255\016\001\130\001\161\001\
    \003\000\255\001\001\000\074\002\149\002\224\002\082\003\196\003\
    \054\004\011\000\016\000\168\004\000\000\026\005\140\005\254\005\
    \216\005\074\006\188\006\250\255\249\255\248\255\247\255\245\255\
    \244\255\046\007\160\007\018\008\239\255\096\008\242\000\240\255\
    \005\000\234\255\031\000\000\000\001\000\029\000\125\000\147\000\
    \026\000\088\000\003\000\003\000\231\255\177\000\153\008\162\000\
    \159\000\192\000\193\000\096\001\153\000\154\000\240\001\188\000\
    \191\000\195\000\193\000\247\000\007\001\251\000\197\000\157\000\
    \001\000\145\001\229\255\163\000\175\000\219\000\156\001\228\255\
    \218\000\177\000\214\000\206\000\203\000\225\000\224\000\226\000\
    \052\001\087\003\227\255\255\000\252\000\036\001\045\001\049\001\
    \102\001\092\003\226\255\045\001\063\001\059\001\043\001\059\001\
    \071\001\165\001\131\001\097\003\225\255\108\003\216\255\078\001\
    \090\001\134\001\113\003\224\255\090\001\111\001\124\001\167\001\
    \201\003\220\255\116\001\156\001\139\001\140\001\198\001\136\001\
    \206\003\223\255\147\001\160\001\205\001\211\003\222\255\154\001\
    \155\001\214\001\222\003\217\255\157\001\160\001\173\001\183\001\
    \165\001\167\001\183\001\227\001\227\003\221\255\199\001\183\001\
    \196\001\202\001\205\001\002\002\059\004\219\255\218\001\216\001\
    \214\001\245\001\239\001\248\001\035\002\064\004\218\255\251\001\
    \031\002\013\002\026\002\030\002\021\002\021\002\013\002\215\255\
    \214\255\238\255\005\000\237\255\141\002\130\002\142\002\142\002\
    \103\002\232\255\069\004\002\000\003\000\183\004\041\002\213\255\
    \212\255\211\255\080\004";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\050\000\049\000\050\000\020\000\019\000\
    \050\000\048\000\050\000\255\255\050\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\047\000\047\000\025\000\
    \022\000\255\255\255\255\047\000\047\000\047\000\047\000\047\000\
    \047\000\008\000\007\000\047\000\255\255\047\000\047\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\012\000\013\000\047\000\255\255\255\255\015\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\025\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\026\000\255\255\255\255\255\255\255\255\027\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\028\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\029\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\030\000\255\255\039\000\255\255\255\255\
    \255\255\255\255\031\000\255\255\255\255\255\255\255\255\255\255\
    \035\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \032\000\255\255\255\255\255\255\255\255\033\000\255\255\255\255\
    \255\255\255\255\038\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\034\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\036\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\037\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\045\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \000\000\255\255\000\000\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\000\000\255\255\255\255\000\000\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\000\000\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\255\255";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\005\000\000\000\003\000\004\000\000\000\245\000\
    \242\000\000\000\245\000\245\000\086\000\086\000\000\000\086\000\
    \086\000\085\000\085\000\000\000\085\000\085\000\000\000\000\000\
    \003\000\060\000\249\000\248\000\000\000\008\000\245\000\000\000\
    \000\000\000\000\000\000\086\000\000\000\000\000\000\000\021\000\
    \085\000\243\000\110\000\244\000\236\000\000\000\000\000\000\000\
    \000\000\000\000\083\000\129\000\010\000\234\000\045\000\044\000\
    \011\000\011\000\019\000\017\000\011\000\011\000\011\000\039\000\
    \020\000\018\000\059\000\032\000\014\000\057\000\011\000\015\000\
    \041\000\055\000\031\000\030\000\052\000\016\000\007\000\056\000\
    \058\000\035\000\036\000\007\000\012\000\006\000\235\000\060\000\
    \096\000\037\000\019\000\017\000\127\000\214\000\206\000\039\000\
    \020\000\018\000\059\000\032\000\014\000\057\000\013\000\015\000\
    \013\000\055\000\031\000\030\000\063\000\016\000\246\000\246\000\
    \065\000\035\000\036\000\009\000\246\000\011\000\241\000\233\000\
    \066\000\037\000\097\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\223\000\013\000\038\000\
    \013\000\196\000\136\000\040\000\061\000\061\000\072\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\071\000\
    \061\000\061\000\061\000\061\000\061\000\070\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\064\000\038\000\
    \053\000\074\000\073\000\040\000\061\000\078\000\061\000\061\000\
    \075\000\061\000\061\000\061\000\077\000\061\000\061\000\067\000\
    \061\000\061\000\068\000\061\000\061\000\061\000\061\000\061\000\
    \069\000\062\000\061\000\061\000\061\000\061\000\076\000\047\000\
    \053\000\131\000\048\000\049\000\050\000\051\000\054\000\108\000\
    \108\000\046\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\028\000\022\000\034\000\122\000\
    \121\000\179\000\026\000\027\000\178\000\107\000\023\000\047\000\
    \033\000\024\000\048\000\049\000\050\000\051\000\054\000\025\000\
    \155\000\046\000\108\000\095\000\095\000\108\000\095\000\095\000\
    \002\000\108\000\156\000\108\000\028\000\022\000\034\000\122\000\
    \121\000\128\000\026\000\027\000\120\000\119\000\023\000\123\000\
    \033\000\024\000\095\000\132\000\133\000\134\000\147\000\025\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\120\000\119\000\108\000\123\000\
    \011\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\139\000\140\000\137\000\141\000\043\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\142\000\124\000\125\000\126\000\138\000\143\000\
    \144\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\124\000\125\000\126\000\145\000\148\000\
    \149\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\232\000\232\000\150\000\232\000\232\000\
    \118\000\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\130\000\130\000\151\000\130\000\130\000\152\000\
    \153\000\172\000\232\000\157\000\167\000\135\000\135\000\160\000\
    \135\000\135\000\109\000\109\000\161\000\109\000\109\000\158\000\
    \159\000\130\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\135\000\165\000\168\000\169\000\
    \170\000\109\000\173\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\107\000\174\000\163\000\
    \175\000\176\000\191\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\180\000\181\000\182\000\
    \184\000\186\000\101\000\187\000\102\000\188\000\104\000\189\000\
    \092\000\092\000\098\000\092\000\092\000\192\000\103\000\193\000\
    \194\000\197\000\198\000\099\000\100\000\105\000\199\000\106\000\
    \162\000\200\000\201\000\202\000\203\000\204\000\108\000\092\000\
    \118\000\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\207\000\208\000\093\000\209\000\108\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\210\000\183\000\211\000\212\000\215\000\216\000\217\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\218\000\219\000\220\000\221\000\093\000\224\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\225\000\226\000\227\000\228\000\
    \229\000\230\000\231\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\235\000\247\000\000\000\
    \000\000\000\000\000\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\091\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\000\000\
    \237\000\238\000\239\000\240\000\000\000\000\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \237\000\238\000\239\000\240\000\000\000\000\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\090\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\089\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\088\000\088\000\000\000\088\000\088\000\
    \146\000\146\000\000\000\146\000\146\000\154\000\154\000\000\000\
    \154\000\154\000\164\000\164\000\000\000\164\000\164\000\000\000\
    \000\000\000\000\088\000\000\000\000\000\166\000\166\000\146\000\
    \166\000\166\000\171\000\171\000\154\000\171\000\171\000\000\000\
    \000\000\164\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\166\000\000\000\000\000\000\000\
    \000\000\171\000\000\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\087\000\087\000\000\000\
    \087\000\087\000\177\000\177\000\000\000\177\000\177\000\185\000\
    \185\000\000\000\185\000\185\000\190\000\190\000\000\000\190\000\
    \190\000\000\000\000\000\000\000\087\000\000\000\000\000\195\000\
    \195\000\177\000\195\000\195\000\205\000\205\000\185\000\205\000\
    \205\000\000\000\000\000\190\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\195\000\000\000\
    \000\000\000\000\000\000\205\000\000\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\052\000\
    \052\000\000\000\052\000\052\000\213\000\213\000\000\000\213\000\
    \213\000\222\000\222\000\000\000\222\000\222\000\250\000\242\000\
    \000\000\250\000\250\000\000\000\000\000\000\000\052\000\000\000\
    \000\000\250\000\242\000\213\000\250\000\250\000\000\000\000\000\
    \222\000\000\000\000\000\000\000\000\000\250\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \250\000\000\000\000\000\000\000\000\000\000\000\000\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\000\000\000\000\000\000\000\000\000\000\000\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\084\000\084\000\000\000\084\000\084\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \245\000\242\000\000\000\245\000\245\000\000\000\000\000\000\000\
    \084\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\245\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\243\000\000\000\244\000\000\000\000\000\000\000\
    \000\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\045\000\045\000\000\000\045\000\045\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\045\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\044\000\044\000\000\000\
    \044\000\044\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\044\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\043\000\
    \043\000\000\000\043\000\043\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\043\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \080\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\043\000\043\000\000\000\043\000\043\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\043\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\000\000\000\000\000\000\
    \000\000\082\000\000\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\043\000\043\000\000\000\
    \043\000\043\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\043\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\000\000\
    \000\000\000\000\000\000\082\000\000\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\056\000\
    \056\000\000\000\056\000\056\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\056\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\000\000\000\000\000\000\000\000\000\000\000\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\058\000\058\000\000\000\058\000\058\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \058\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\060\000\060\000\000\000\060\000\060\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\060\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\093\000\000\000\000\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\094\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\000\000\000\000\000\000\000\000\093\000\
    \000\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\117\000\111\000\000\000\000\000\000\000\
    \000\000\115\000\116\000\000\000\000\000\112\000\000\000\000\000\
    \113\000\000\000\000\000\000\000\000\000\000\000\114\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\117\000\111\000\000\000\000\000\000\000\
    \000\000\115\000\116\000\000\000\000\000\112\000\000\000\000\000\
    \113\000\000\000\000\000\000\000\000\000\000\000\114\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\255\255\000\000\000\000\255\255\005\000\
    \005\000\255\255\005\000\005\000\073\000\073\000\255\255\073\000\
    \073\000\074\000\074\000\255\255\074\000\074\000\255\255\255\255\
    \000\000\066\000\243\000\244\000\255\255\000\000\005\000\255\255\
    \255\255\255\255\255\255\073\000\255\255\255\255\255\255\010\000\
    \074\000\005\000\107\000\005\000\234\000\255\255\255\255\255\255\
    \255\255\255\255\076\000\128\000\000\000\007\000\018\000\019\000\
    \027\000\028\000\010\000\010\000\030\000\031\000\032\000\033\000\
    \010\000\010\000\014\000\023\000\010\000\015\000\038\000\010\000\
    \040\000\016\000\024\000\025\000\051\000\010\000\054\000\055\000\
    \057\000\034\000\035\000\000\000\000\000\000\000\007\000\059\000\
    \064\000\036\000\010\000\010\000\106\000\099\000\100\000\033\000\
    \010\000\010\000\014\000\023\000\010\000\015\000\012\000\010\000\
    \012\000\016\000\024\000\025\000\009\000\010\000\003\000\004\000\
    \009\000\034\000\035\000\000\000\005\000\000\000\006\000\008\000\
    \009\000\036\000\096\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\098\000\012\000\037\000\
    \012\000\101\000\104\000\039\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\037\000\
    \046\000\009\000\009\000\039\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\017\000\
    \046\000\105\000\047\000\048\000\049\000\050\000\053\000\116\000\
    \117\000\017\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\021\000\021\000\022\000\111\000\
    \112\000\102\000\021\000\021\000\102\000\109\000\021\000\017\000\
    \022\000\021\000\047\000\048\000\049\000\050\000\053\000\021\000\
    \103\000\017\000\119\000\094\000\094\000\120\000\094\000\094\000\
    \000\000\121\000\103\000\126\000\021\000\021\000\022\000\111\000\
    \112\000\127\000\021\000\021\000\113\000\114\000\021\000\122\000\
    \022\000\021\000\094\000\131\000\132\000\133\000\137\000\021\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\113\000\114\000\109\000\122\000\
    \029\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\138\000\139\000\136\000\140\000\042\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\141\000\123\000\124\000\125\000\136\000\142\000\
    \143\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\123\000\124\000\125\000\144\000\147\000\
    \148\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\062\000\062\000\149\000\062\000\062\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\129\000\129\000\150\000\129\000\129\000\151\000\
    \152\000\155\000\062\000\156\000\158\000\134\000\134\000\159\000\
    \134\000\134\000\063\000\063\000\160\000\063\000\063\000\157\000\
    \157\000\129\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\134\000\162\000\167\000\168\000\
    \169\000\063\000\172\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\063\000\173\000\161\000\
    \174\000\175\000\178\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\179\000\180\000\181\000\
    \182\000\183\000\063\000\186\000\063\000\187\000\063\000\188\000\
    \065\000\065\000\063\000\065\000\065\000\191\000\063\000\192\000\
    \193\000\196\000\197\000\063\000\063\000\063\000\198\000\063\000\
    \161\000\199\000\200\000\201\000\202\000\203\000\063\000\065\000\
    \118\000\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\206\000\207\000\065\000\208\000\118\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\209\000\182\000\210\000\211\000\214\000\215\000\216\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\217\000\218\000\219\000\220\000\065\000\223\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\224\000\225\000\226\000\227\000\
    \228\000\229\000\230\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\240\000\246\000\255\255\
    \255\255\255\255\255\255\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\255\255\
    \236\000\237\000\238\000\239\000\255\255\255\255\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \236\000\237\000\238\000\239\000\255\255\255\255\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\070\000\070\000\255\255\070\000\070\000\
    \145\000\145\000\255\255\145\000\145\000\153\000\153\000\255\255\
    \153\000\153\000\163\000\163\000\255\255\163\000\163\000\255\255\
    \255\255\255\255\070\000\255\255\255\255\165\000\165\000\145\000\
    \165\000\165\000\170\000\170\000\153\000\170\000\170\000\255\255\
    \255\255\163\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\165\000\255\255\255\255\255\255\
    \255\255\170\000\255\255\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\071\000\071\000\255\255\
    \071\000\071\000\176\000\176\000\255\255\176\000\176\000\184\000\
    \184\000\255\255\184\000\184\000\189\000\189\000\255\255\189\000\
    \189\000\255\255\255\255\255\255\071\000\255\255\255\255\194\000\
    \194\000\176\000\194\000\194\000\204\000\204\000\184\000\204\000\
    \204\000\255\255\255\255\189\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\194\000\255\255\
    \255\255\255\255\255\255\204\000\255\255\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\072\000\
    \072\000\255\255\072\000\072\000\212\000\212\000\255\255\212\000\
    \212\000\221\000\221\000\255\255\221\000\221\000\242\000\242\000\
    \255\255\242\000\242\000\255\255\255\255\255\255\072\000\255\255\
    \255\255\250\000\250\000\212\000\250\000\250\000\255\255\255\255\
    \221\000\255\255\255\255\255\255\255\255\242\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \250\000\255\255\255\255\255\255\255\255\255\255\255\255\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\255\255\255\255\255\255\255\255\255\255\255\255\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\075\000\075\000\255\255\075\000\075\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \245\000\245\000\255\255\245\000\245\000\255\255\255\255\255\255\
    \075\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\245\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\245\000\255\255\245\000\255\255\255\255\255\255\
    \255\255\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\077\000\077\000\255\255\077\000\077\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\077\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\078\000\078\000\255\255\
    \078\000\078\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\078\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\079\000\
    \079\000\255\255\079\000\079\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\079\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\079\000\
    \079\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\081\000\081\000\255\255\081\000\081\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\081\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\255\255\255\255\255\255\
    \255\255\081\000\255\255\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\082\000\082\000\255\255\
    \082\000\082\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\082\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\255\255\
    \255\255\255\255\255\255\082\000\255\255\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\089\000\
    \089\000\255\255\089\000\089\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\089\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\255\255\255\255\255\255\255\255\255\255\255\255\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\090\000\090\000\255\255\090\000\090\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \090\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \090\000\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\090\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\090\000\091\000\091\000\255\255\091\000\091\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\091\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\093\000\255\255\255\255\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\255\255\255\255\255\255\255\255\093\000\
    \255\255\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\110\000\110\000\255\255\255\255\255\255\
    \255\255\110\000\110\000\255\255\255\255\110\000\255\255\255\255\
    \110\000\255\255\255\255\255\255\255\255\255\255\110\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\110\000\110\000\255\255\255\255\255\255\
    \255\255\110\000\110\000\255\255\255\255\110\000\255\255\255\255\
    \110\000\255\255\255\255\255\255\255\255\255\255\110\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_base_code =
   "";
  Lexing.lex_backtrk_code =
   "";
  Lexing.lex_default_code =
   "";
  Lexing.lex_trans_code =
   "";
  Lexing.lex_check_code =
   "";
  Lexing.lex_code =
   "";
}

let rec main lexbuf =
   __ocaml_lex_main_rec lexbuf 0
and __ocaml_lex_main_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 160 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      let s = Lexing.lexeme lexbuf in
      Char (String.sub s 1 1)
    )
# 814 "ocamldoc/odoc_text_lexer.ml"

  | 1 ->
# 167 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) then
        Char (Lexing.lexeme lexbuf)
      else begin
        if !ele_ref_mode then
          ele_ref_mode := false;
        END
      end
    )
# 829 "ocamldoc/odoc_text_lexer.ml"

  | 2 ->
# 179 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        let s = Lexing.lexeme lexbuf in
        try
          (* check if the "{..." or html_title mark was used. *)
          if s.[0] = '<' then
            let (n, l) = (2, (String.length s - 3)) in
            let s2 = String.sub s n l in
            Title (int_of_string s2, None)
          else
            let (n, l) = (1, (String.length s - 2)) in
            let s2 = String.sub s n l in
            try
              let i = String.index s2 ':' in
              let s_n = String.sub s2 0 i in
              let s_label = String.sub s2 (i+1) (l-i-1) in
              Title (int_of_string s_n, Some s_label)
            with
              Not_found ->
                Title (int_of_string s2, None)
        with
          _ ->
            Title (1, None)
    )
# 861 "ocamldoc/odoc_text_lexer.ml"

  | 3 ->
# 208 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        BOLD
    )
# 873 "ocamldoc/odoc_text_lexer.ml"

  | 4 ->
# 217 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        ITALIC
    )
# 885 "ocamldoc/odoc_text_lexer.ml"

  | 5 ->
# 226 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        LINK
    )
# 897 "ocamldoc/odoc_text_lexer.ml"

  | 6 ->
# 235 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        EMP
    )
# 909 "ocamldoc/odoc_text_lexer.ml"

  | 7 ->
# 244 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        SUPERSCRIPT
    )
# 921 "ocamldoc/odoc_text_lexer.ml"

  | 8 ->
# 253 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        SUBSCRIPT
    )
# 933 "ocamldoc/odoc_text_lexer.ml"

  | 9 ->
# 262 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        CENTER
    )
# 945 "ocamldoc/odoc_text_lexer.ml"

  | 10 ->
# 271 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        LEFT
    )
# 957 "ocamldoc/odoc_text_lexer.ml"

  | 11 ->
# 280 "ocamldoc/odoc_text_lexer.mll"
     (
      incr_cpts lexbuf ;
       if !verb_mode || !target_mode || !code_pre_mode
           || (!open_brackets >= 1) || !ele_ref_mode then
         Char (Lexing.lexeme lexbuf)
       else
         RIGHT
     )
# 969 "ocamldoc/odoc_text_lexer.ml"

  | 12 ->
# 289 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        LIST
    )
# 981 "ocamldoc/odoc_text_lexer.ml"

  | 13 ->
# 298 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        ENUM
    )
# 993 "ocamldoc/odoc_text_lexer.ml"

  | 14 ->
# 307 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        ITEM
    )
# 1005 "ocamldoc/odoc_text_lexer.ml"

  | 15 ->
# 316 "ocamldoc/odoc_text_lexer.mll"
   (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        (
         let s = Lexing.lexeme lexbuf in
         let fmt =
           let p1 = String.index s '%' in
           let p2 = String.index s ':' in
           String.sub s (p1 + 1) (p2 - p1 - 1)
         in
         target_mode := true;
         Target fmt
        )
    )
# 1026 "ocamldoc/odoc_text_lexer.ml"

  | 16 ->
# 334 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        (
         target_mode := true;
         LATEX
        )
    )
# 1041 "ocamldoc/odoc_text_lexer.ml"

  | 17 ->
# 346 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || (!open_brackets >= 1) || !code_pre_mode ||
        !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        (
         target_mode := false;
         END_TARGET
        )
    )
# 1056 "ocamldoc/odoc_text_lexer.ml"

  | 18 ->
# 358 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      Char (Lexing.lexeme lexbuf)
    )
# 1064 "ocamldoc/odoc_text_lexer.ml"

  | 19 ->
# 364 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        if !open_brackets <= 0 then
          (
           open_brackets := 1;
           CODE
          )
        else
          (
           incr open_brackets;
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1084 "ocamldoc/odoc_text_lexer.ml"

  | 20 ->
# 381 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        if !open_brackets > 1 then
          (
           decr open_brackets;
           Char "]"
          )
        else
          (
           open_brackets := 0;
           END_CODE
          )
    )
# 1104 "ocamldoc/odoc_text_lexer.ml"

  | 21 ->
# 399 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      Char (Lexing.lexeme lexbuf)
    )
# 1112 "ocamldoc/odoc_text_lexer.ml"

  | 22 ->
# 405 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        (
         code_pre_mode := true;
         CODE_PRE
        )
    )
# 1126 "ocamldoc/odoc_text_lexer.ml"

  | 23 ->
# 416 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        if !open_brackets >= 1 then
          (
           lexbuf.Lexing.lex_curr_pos <- lexbuf.Lexing.lex_curr_pos - 1;
           lexbuf.Lexing.lex_curr_p <-
             { lexbuf.Lexing.lex_curr_p with
               pos_cnum = lexbuf.Lexing.lex_curr_p.pos_cnum - 1
             } ;
           decr char_number ;
           if !open_brackets > 1 then
             (
              decr open_brackets;
              Char "]"
             )
           else
             (
              open_brackets := 0;
              END_CODE
             )
          )
        else
          if !code_pre_mode then
            (
             code_pre_mode := false;
             END_CODE_PRE
            )
          else
            Char (Lexing.lexeme lexbuf)
    )
# 1163 "ocamldoc/odoc_text_lexer.ml"

  | 24 ->
# 451 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      Char (Lexing.lexeme lexbuf)
    )
# 1171 "ocamldoc/odoc_text_lexer.ml"

  | 25 ->
# 457 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           ELE_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1190 "ocamldoc/odoc_text_lexer.ml"

  | 26 ->
# 475 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           VAL_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1209 "ocamldoc/odoc_text_lexer.ml"

  | 27 ->
# 492 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           TYP_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1228 "ocamldoc/odoc_text_lexer.ml"

  | 28 ->
# 509 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           EXT_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1247 "ocamldoc/odoc_text_lexer.ml"

  | 29 ->
# 526 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           EXC_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1266 "ocamldoc/odoc_text_lexer.ml"

  | 30 ->
# 543 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           MOD_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1285 "ocamldoc/odoc_text_lexer.ml"

  | 31 ->
# 560 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           MODT_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1304 "ocamldoc/odoc_text_lexer.ml"

  | 32 ->
# 577 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           CLA_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1323 "ocamldoc/odoc_text_lexer.ml"

  | 33 ->
# 594 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           CLT_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1342 "ocamldoc/odoc_text_lexer.ml"

  | 34 ->
# 611 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           ATT_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1361 "ocamldoc/odoc_text_lexer.ml"

  | 35 ->
# 628 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           MET_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1380 "ocamldoc/odoc_text_lexer.ml"

  | 36 ->
# 645 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           SEC_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1399 "ocamldoc/odoc_text_lexer.ml"

  | 37 ->
# 661 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           RECF_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1418 "ocamldoc/odoc_text_lexer.ml"

  | 38 ->
# 677 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           CONST_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1437 "ocamldoc/odoc_text_lexer.ml"

  | 39 ->
# 693 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          (
           ele_ref_mode := true;
           MOD_LIST_REF
          )
        else
          (
           Char (Lexing.lexeme lexbuf)
          )
    )
# 1456 "ocamldoc/odoc_text_lexer.ml"

  | 40 ->
# 710 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode || !open_brackets >= 1 then
        Char (Lexing.lexeme lexbuf)
      else
        if not !ele_ref_mode then
          INDEX_LIST
        else
          Char (Lexing.lexeme lexbuf)
    )
# 1470 "ocamldoc/odoc_text_lexer.ml"

  | 41 ->
# 722 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !target_mode || (!open_brackets >= 1) || !code_pre_mode || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        (
         verb_mode := true;
         VERB
        )
    )
# 1484 "ocamldoc/odoc_text_lexer.ml"

  | 42 ->
# 733 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !target_mode || (!open_brackets >= 1) || !code_pre_mode || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        (
         verb_mode := false;
         END_VERB
        )
    )
# 1498 "ocamldoc/odoc_text_lexer.ml"

  | 43 ->
# 745 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !target_mode || (!open_brackets >= 1) || !code_pre_mode
          || !ele_ref_mode || !verb_mode then
        Char (Lexing.lexeme lexbuf)
      else if !shortcut_list_mode then
        (
         SHORTCUT_LIST_ITEM
        )
      else
        (
         shortcut_list_mode := true;
         BEGIN_SHORTCUT_LIST_ITEM
        )
    )
# 1517 "ocamldoc/odoc_text_lexer.ml"

  | 44 ->
# 762 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !target_mode || (!open_brackets >= 1) || !code_pre_mode
         || !ele_ref_mode || !verb_mode then
        Char (Lexing.lexeme lexbuf)
      else if !shortcut_list_mode then
        SHORTCUT_ENUM_ITEM
      else
        (
         shortcut_list_mode := true;
         BEGIN_SHORTCUT_ENUM_ITEM
        )
    )
# 1534 "ocamldoc/odoc_text_lexer.ml"

  | 45 ->
# 776 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      lexbuf.Lexing.lex_curr_pos <- lexbuf.Lexing.lex_curr_pos - 1;
      lexbuf.Lexing.lex_curr_p <-
        { lexbuf.Lexing.lex_curr_p with
          pos_cnum = lexbuf.Lexing.lex_curr_p.pos_cnum - 1 ;
        } ;
      decr line_number ;
      if !shortcut_list_mode then
        (
         shortcut_list_mode := false;
                        (* go back one char to reuse the last '\n', so we can
                           restart another shortcut-list with a single blank line,
                           and not two.*)
         END_SHORTCUT_LIST
        )
      else
        if !target_mode || (!open_brackets >= 1) || !code_pre_mode || !ele_ref_mode || !verb_mode then
          Char (Lexing.lexeme lexbuf)
        else
          BLANK_LINE
    )
# 1560 "ocamldoc/odoc_text_lexer.ml"

  | 46 ->
# 799 "ocamldoc/odoc_text_lexer.mll"
                ( EOF )
# 1565 "ocamldoc/odoc_text_lexer.ml"

  | 47 ->
# 802 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !verb_mode || !target_mode || !code_pre_mode ||
        (!open_brackets >= 1) || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        let s = Lexing.lexeme lexbuf in
        let len = String.length s in
        (* remove this starting '{' *)
        let tag = Odoc_misc.no_blanks (String.sub s 1 (len - 1)) in
        CUSTOM tag
    )
# 1581 "ocamldoc/odoc_text_lexer.ml"

  | 48 ->
# 816 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      if !target_mode || (!open_brackets >= 1) || !code_pre_mode || !ele_ref_mode then
        Char (Lexing.lexeme lexbuf)
      else
        LBRACE
    )
# 1592 "ocamldoc/odoc_text_lexer.ml"

  | 49 ->
# 823 "ocamldoc/odoc_text_lexer.mll"
       ( main lexbuf )
# 1597 "ocamldoc/odoc_text_lexer.ml"

  | 50 ->
# 825 "ocamldoc/odoc_text_lexer.mll"
    (
      incr_cpts lexbuf ;
      Char (Lexing.lexeme lexbuf)
    )
# 1605 "ocamldoc/odoc_text_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_main_rec lexbuf __ocaml_lex_state

;;

